<?php
/**
 * Plugin Name: シンプルセミナー一覧
 * Description: 日付順に並ぶシンプルなセミナー一覧。自動的に終了イベントを分離表示
 * Version: 1.0
 */

// カスタム投稿タイプの登録
function register_simple_seminar_post_type() {
    register_post_type('simple_seminar', array(
        'labels' => array(
            'name' => 'セミナー',
            'singular_name' => 'セミナー',
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title'),
        'menu_icon' => 'dashicons-calendar-alt',
    ));
}
add_action('init', 'register_simple_seminar_post_type');

// 日付入力フィールドの追加
function add_seminar_date_meta_box() {
    add_meta_box(
        'seminar_date',
        'セミナー日付',
        'render_seminar_date_meta_box',
        'simple_seminar',
        'normal'
    );
}
add_action('add_meta_boxes', 'add_seminar_date_meta_box');

function render_seminar_date_meta_box($post) {
    $date = get_post_meta($post->ID, '_seminar_date', true);
    wp_nonce_field('save_seminar_date', 'seminar_date_nonce');
    echo '<input type="date" name="seminar_date" value="' . esc_attr($date) . '" />';
}

// 日付の保存
function save_seminar_date($post_id) {
    if (!isset($_POST['seminar_date_nonce']) ||
        !wp_verify_nonce($_POST['seminar_date_nonce'], 'save_seminar_date')) {
        return;
    }
    if (isset($_POST['seminar_date'])) {
        update_post_meta($post_id, '_seminar_date', sanitize_text_field($_POST['seminar_date']));
    }
}
add_action('save_post_simple_seminar', 'save_seminar_date');

// ショートコード [seminar_list] で表示
function display_seminar_list() {
    $today = current_time('Y-m-d');
    
    // 日本語曜日の配列
    $weekdays = array('日', '月', '火', '水', '木', '金', '土');
    
    // 開催予定のセミナー
    $upcoming = new WP_Query(array(
        'post_type' => 'simple_seminar',
        'posts_per_page' => -1,
        'meta_key' => '_seminar_date',
        'orderby' => 'meta_value',
        'order' => 'ASC',
        'meta_query' => array(
            array(
                'key' => '_seminar_date',
                'value' => $today,
                'compare' => '>=',
                'type' => 'DATE'
            )
        )
    ));
    
    // 終了したセミナー
    $past = new WP_Query(array(
        'post_type' => 'simple_seminar',
        'posts_per_page' => -1,
        'meta_key' => '_seminar_date',
        'orderby' => 'meta_value',
        'order' => 'DESC',
        'meta_query' => array(
            array(
                'key' => '_seminar_date',
                'value' => $today,
                'compare' => '<',
                'type' => 'DATE'
            )
        )
    ));
    
    ob_start();
    ?>
    <style>
        .seminar-list ul {
            list-style: none;
            padding-left: 0;
        }
    </style>
    <div class="seminar-list">
        <h3>開催予定</h3>
        <?php if ($upcoming->have_posts()) : ?>
            <ul>
            <?php while ($upcoming->have_posts()) : $upcoming->the_post();
                $date = get_post_meta(get_the_ID(), '_seminar_date', true);
                $timestamp = strtotime($date);
                $formatted_date = date('n月j日', $timestamp) . '(' . $weekdays[date('w', $timestamp)] . ')';
            ?>
                <li><?php echo $formatted_date; ?> <?php the_title(); ?></li>
            <?php endwhile; ?>
            </ul>
        <?php else : ?>
            <p>現在、開催予定のセミナーはありません。</p>
        <?php endif; wp_reset_postdata(); ?>
        
        <?php if ($past->have_posts()) : ?>
            <h3>終了しました</h3>
            <ul>
            <?php while ($past->have_posts()) : $past->the_post();
                $date = get_post_meta(get_the_ID(), '_seminar_date', true);
                $timestamp = strtotime($date);
                $formatted_date = date('n月j日', $timestamp) . '(' . $weekdays[date('w', $timestamp)] . ')';
            ?>
                <li><?php echo $formatted_date; ?> <?php the_title(); ?></li>
            <?php endwhile; ?>
            </ul>
        <?php endif; wp_reset_postdata(); ?>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('seminar_list', 'display_seminar_list');

// 管理画面にセミナー日付カラムを追加
function add_seminar_date_column($columns) {
    $new_columns = array();
    foreach ($columns as $key => $value) {
        $new_columns[$key] = $value;
        if ($key === 'title') {
            $new_columns['seminar_date'] = 'セミナー日付';
        }
    }
    return $new_columns;
}
add_filter('manage_simple_seminar_posts_columns', 'add_seminar_date_column');

// カラムの内容を表示(JavaScriptで読み取れるようにspan要素でラップ)
function display_seminar_date_column($column, $post_id) {
    if ($column === 'seminar_date') {
        $date = get_post_meta($post_id, '_seminar_date', true);
        if ($date) {
            $weekdays = array('日', '月', '火', '水', '木', '金', '土');
            $timestamp = strtotime($date);
            echo '<span class="seminar_date">' . date('Y年n月j日', $timestamp) . '(' . $weekdays[date('w', $timestamp)] . ')</span>';
        } else {
            echo '—';
        }
    }
}
add_action('manage_simple_seminar_posts_custom_column', 'display_seminar_date_column', 10, 2);

// カラムをソート可能にする
function make_seminar_date_sortable($columns) {
    $columns['seminar_date'] = 'seminar_date';
    return $columns;
}
add_filter('manage_edit-simple_seminar_sortable_columns', 'make_seminar_date_sortable');

// ソート処理
function seminar_date_orderby($query) {
    if (!is_admin() || !$query->is_main_query()) {
        return;
    }
    if ($query->get('orderby') === 'seminar_date') {
        $query->set('meta_key', '_seminar_date');
        $query->set('orderby', 'meta_value');
    }
}
add_action('pre_get_posts', 'seminar_date_orderby');

// クイック編集にセミナー日付を追加
function add_quick_edit_seminar_date($column_name, $post_type) {
    if ($post_type !== 'simple_seminar' || $column_name !== 'seminar_date') {
        return;
    }
    ?>
    <fieldset class="inline-edit-col-right">
        <div class="inline-edit-col">
            <label>
                <span class="title">セミナー日付</span>
                <span class="input-text-wrap">
                    <input type="date" name="seminar_date" value="">
                </span>
            </label>
        </div>
    </fieldset>
    <?php
}
add_action('quick_edit_custom_box', 'add_quick_edit_seminar_date', 10, 2);

// クイック編集のJavaScript(既存の日付を入力欄にセット)
function enqueue_quick_edit_script() {
    global $current_screen;
    if ($current_screen->post_type !== 'simple_seminar') {
        return;
    }
    ?>
    <script type="text/javascript">
    jQuery(document).ready(function($) {
        var $inline_editor = inlineEditPost.edit;
        inlineEditPost.edit = function(id) {
            $inline_editor.apply(this, arguments);
            var post_id = 0;
            if (typeof(id) == 'object') {
                post_id = parseInt(this.getId(id));
            }
            if (post_id > 0) {
                var $row = $('#post-' + post_id);
                var seminar_date = $row.find('.seminar_date').text();
                // 日付を抽出 (例: 2026年1月15日(水) → 2026-01-15)
                var match = seminar_date.match(/(\d{4})年(\d{1,2})月(\d{1,2})日/);
                if (match) {
                    var year = match[1];
                    var month = match[2].padStart(2, '0');
                    var day = match[3].padStart(2, '0');
                    var date_value = year + '-' + month + '-' + day;
                    $('input[name="seminar_date"]').val(date_value);
                }
            }
        };
    });
    </script>
    <?php
}
add_action('admin_footer', 'enqueue_quick_edit_script');

// クイック編集の保存処理
function save_quick_edit_seminar_date($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    if (isset($_POST['seminar_date']) && $_POST['seminar_date'] !== '') {
        update_post_meta($post_id, '_seminar_date', sanitize_text_field($_POST['seminar_date']));
    }
}
add_action('save_post_simple_seminar', 'save_quick_edit_seminar_date');
